---
title: "Docker Compose : introduction simple"
course: "docker"
slug: "7-docker-compose-intro-simple"
order: 7
createdAt: 2025-09-24
updatedAt: 2025-09-24
---

Quand un projet utilise plusieurs conteneurs (par exemple : une app web + une base de données), gérer tout à la main devient vite compliqué. **Docker Compose** permet de définir et lancer plusieurs services Docker avec un seul fichier.

---

## Qu'est-ce que Docker Compose ?

**Docker Compose** est un outil qui permet de définir et gérer des applications multi-conteneurs à l'aide d'un fichier YAML. Au lieu de lancer plusieurs commandes `docker run`, vous décrivez votre infrastructure complète dans un fichier `compose.yml`.

> [!info] Avantage principal
> Un seul fichier pour documenter et lancer toute votre architecture !

---

## Qu'est-ce que compose.yml ?

C'est un fichier YAML dans lequel tu décris :

- Les **services** (conteneurs)
- Les **volumes** pour la persistance
- Les **réseaux** pour la communication
- Les **variables d'environnement**
- Les **ports** à exposer

> [!tip] Changement important 2025
> Depuis Docker Compose v2, **plus besoin de spécifier la version** ! La ligne `version: '3.8'` est maintenant obsolète.

---

## Exemple simple : App web + Base de données

### Structure du projet

```
mon-projet/
├── compose.yml
├── app/
│   ├── Dockerfile
│   └── main.py
└── .env
```

### compose.yml

```yaml
services:
  web:
    build: ./app
    ports:
      - "5000:5000"
    volumes:
      - ./app:/app
    environment:
      - DATABASE_URL=postgresql://user:pass@db:5432/mydb
    depends_on:
      - db

  db:
    image: postgres:15
    environment:
      POSTGRES_USER: user
      POSTGRES_PASSWORD: pass
      POSTGRES_DB: mydb
    volumes:
      - postgres_data:/var/lib/postgresql/data
    ports:
      - "5432:5432"

volumes:
  postgres_data:
```

---

## Commandes de base

> [!warning] Important
> Utilisez `docker compose` (avec espace) et non `docker-compose` (avec tiret) depuis 2025 !

### Lancer tous les services

```bash
docker compose up
```

**Options utiles :**

- `-d` : en arrière-plan
- `--build` : force la reconstruction des images

### Arrêter et supprimer les conteneurs

```bash
docker compose down
```

### Voir les services en cours

```bash
docker compose ps
```

### Voir les logs

```bash
docker compose logs
docker compose logs web        # logs d'un service spécifique
docker compose logs -f        # suivi en temps réel
```

### Reconstruire les images

```bash
docker compose build
docker compose up --build
```

---

## Exemple complet avec plusieurs services

```yaml
services:
  # Application web
  frontend:
    build:
      context: ./frontend
      dockerfile: Dockerfile
    ports:
      - "3000:3000"
    volumes:
      - ./frontend:/app
      - /app/node_modules
    environment:
      - REACT_APP_API_URL=http://localhost:8000

  # API Backend
  backend:
    build: ./backend
    ports:
      - "8000:8000"
    volumes:
      - ./backend:/app
    environment:
      - DATABASE_URL=postgresql://postgres:password@database:5432/myapp
      - REDIS_URL=redis://redis:6379
    depends_on:
      - database
      - redis

  # Base de données
  database:
    image: postgres:15
    environment:
      POSTGRES_DB: myapp
      POSTGRES_USER: postgres
      POSTGRES_PASSWORD: password
    volumes:
      - db_data:/var/lib/postgresql/data
    ports:
      - "5432:5432"

  # Cache Redis
  redis:
    image: redis:7-alpine
    ports:
      - "6379:6379"
    volumes:
      - redis_data:/data

  # Nginx reverse proxy
  nginx:
    image: nginx:alpine
    ports:
      - "80:80"
    volumes:
      - ./nginx.conf:/etc/nginx/nginx.conf
    depends_on:
      - frontend
      - backend

volumes:
  db_data:
  redis_data:

networks:
  default:
    name: myapp_network
```

---

## Gestion des variables d'environnement

### Fichier .env

```env
# Base de données
POSTGRES_DB=myapp
POSTGRES_USER=postgres
POSTGRES_PASSWORD=supersecret

# Application
APP_ENV=development
DEBUG=true
SECRET_KEY=your-secret-key
```

### Utilisation dans compose.yml

```yaml
services:
  app:
    image: mon-app
    environment:
      - DATABASE_URL=postgresql://${POSTGRES_USER}:${POSTGRES_PASSWORD}@db:5432/${POSTGRES_DB}
      - APP_ENV=${APP_ENV}
    env_file:
      - .env
```

> [!tip] Bonne pratique
> Utilisez toujours un fichier `.env` pour les secrets et ne le committez jamais !

---

## Réseaux et communication

### Communication entre services

Les services peuvent communiquer entre eux en utilisant leurs **noms** comme hostnames :

```python
# Dans votre application Python
import psycopg2

# Le hostname 'db' correspond au service 'db' du compose.yml
conn = psycopg2.connect(
    host="db",
    database="myapp",
    user="postgres",
    password="password"
)
```

### Réseaux personnalisés

```yaml
services:
  web:
    networks:
      - frontend
      - backend
  
  db:
    networks:
      - backend

networks:
  frontend:
  backend:
    internal: true  # Réseau interne uniquement
```

---

## Gestion des volumes

### Types de volumes dans Compose

```yaml
services:
  app:
    volumes:
      # Bind mount (développement)
      - ./src:/app/src
      
      # Named volume (production)
      - app_data:/app/data
      
      # Volume temporaire
      - /tmp
      
      # Volume en lecture seule
      - ./config:/etc/app:ro

volumes:
  app_data:
    driver: local
```

---

## Profils pour différents environnements

```yaml
services:
  app:
    image: mon-app
    
  db:
    image: postgres:15
    profiles:
      - production
      
  redis:
    image: redis:7
    profiles:
      - production
      
  mailhog:
    image: mailhog/mailhog
    profiles:
      - development
    ports:
      - "8025:8025"
```

**Utilisation :**

```bash
# Développement (sans db et redis)
docker compose up

# Production
docker compose --profile production up

# Développement avec mailhog
docker compose --profile development up
```

---

## Commandes utiles avancées

### Exécuter une commande dans un service

```bash
docker compose exec web bash
docker compose exec db psql -U postgres myapp
```

### Scaler des services

```bash
docker compose up --scale web=3
```

### Voir la configuration finale

```bash
docker compose config
```

### Arrêter un service spécifique

```bash
docker compose stop web
docker compose start web
```

### Rebuild un service spécifique

```bash
docker compose build web
docker compose up -d --no-deps web
```

---

## Bonnes pratiques

### Structure des fichiers

```
projet/
├── compose.yml
├── compose.override.yml    # Surcharges pour le dev
├── compose.prod.yml       # Configuration production
├── .env                   # Variables d'environnement
├── .env.example          # Exemple sans secrets
└── services/
    ├── frontend/
    ├── backend/
    └── nginx/
```

### Multi-environnements

```bash
# Développement
docker compose up

# Production
docker compose -f compose.yml -f compose.prod.yml up

# Test
docker compose -f compose.yml -f compose.test.yml up
```

### Santé des services

```yaml
services:
  web:
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:8000/health"]
      interval: 30s
      timeout: 10s
      retries: 3
      start_period: 40s
```

---

## Avantages de Docker Compose

**Simplicité** : Un seul fichier pour documenter ton architecture

**Reproductibilité** : Facile à démarrer tout un projet sur n'importe quelle machine

**Isolation** : Chaque service est isolé mais connecté

**Gestion des dépendances** : `depends_on` gère l'ordre de démarrage

**Développement** : Rechargement à chaud avec les bind mounts

**Production** : Volumes persistants et configuration robuste

---

## À retenir

- **Plus de `version:`** dans les fichiers compose.yml modernes
- **`docker compose`** (avec espace) au lieu de `docker-compose`
- **services:** définit les conteneurs
- **volumes:** permet la persistance des données
- **networks:** pour la communication
- **depends_on:** gère les dépendances
- **env_file:** pour les variables d'environnement
- **Communication** entre services par leur nom
- **Profils** pour différents environnements

---

## Liens vers les autres parties

Vous avez atteint la fin actuelle de ce cours, d'autres parties avancées arriverons prochainement

---

## Tags

#docker #docker-compose #multi-conteneurs #orchestration #yaml #services #microservices